-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

THModManager = {}
local THModManager_mt = Class(THModManager)
THModManager.CLASS_NAME = "THModManager"
local function initScript()
    local self = THModManager.new()
    if self ~= nil then
        _G.g_thModManager = self
        self:initActiveMods()
    end
end
function THModManager.new(customMt)
    customMt = customMt or THModManager_mt
    if THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt) then
        local self = setmetatable({}, customMt)
        self.activeMods = {
            array = {},
            byName = {},
            byIndex = {}
        }
        return self
    end
end
function THModManager.initActiveMods(self)
    local modsArray = g_mpLoadingScreen.missionDynamicInfo.mods
    for modIndex, modInfo in ipairs(modsArray) do
        local modName = THUtils.getNoNil(modInfo.modName, "")
        if modName ~= "" then
            local modNameUpper = string.upper(modName)
            local modEnv = THUtils.getModEnvironment(modName)
            if modEnv == nil then
                THUtils.errorMsg(false, "Cannot find mod environment: %s", modName)
            elseif self.activeMods.byName[modNameUpper] == nil then
                local modData = {
                    name = modName,
                    desc = modInfo,
                    path = modInfo.modDir,
                    file = modInfo.modFile,
                    title = modInfo.title,
                    descVersion = modInfo.modDescVersion,
                    environment = modEnv
                }
                local modVersion = modInfo.version
                if type(modVersion) == THValueType.STRING then
                    local _, _, versionText = string.find(modVersion, "([0-9.]+)")
                    local versionData = THUtils.splitString(versionText, ".", true)
                    if versionData == nil then
                        versionData = {}
                    end
                    modData.version = {
                        text = modVersion,
                        major = THUtils.toNumber(versionData[1], true) or 0,
                        minor = THUtils.toNumber(versionData[2], true) or 0,
                        revision = THUtils.toNumber(versionData[3], true) or 0,
                        build = THUtils.toNumber(versionData[4], true) or 0
                    }
                else
                    modData.version = {
                        text = "0.0.0.0",
                        major = 0,
                        minor = 0,
                        revision = 0,
                        build = 0
                    }
                end
                table.insert(self.activeMods.array, modData)
                self.activeMods.byIndex[modIndex] = modData
                self.activeMods.byName[modNameUpper] = modData
            end
        end
    end
end
function THModManager.getActiveMod(self, modId, verbose)
    local idVarType = type(modId)
    local modData = nil
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    if idVarType == THValueType.STRING then
        modData = self.activeMods.byName[modId:upper()]
    elseif idVarType == THValueType.NUMBER then
        modData = self.activeMods.byIndex[modId]
    elseif modData ~= nil then
        verbose = true
    end
    if modData == nil and verbose then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "modId", modId)
    end
    return modData
end
function THModManager.getActiveMods(self)
    return self.activeMods.array, #self.activeMods.array
end
function THModManager.getActiveModByFilename(self, filename)
    if filename ~= nil
        and THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
    then
        local activeModsArray, numActiveMods = self:getActiveMods()
        if activeModsArray ~= nil and numActiveMods > 0 then
            for modIndex = 1, numActiveMods do
                local modData = activeModsArray[modIndex]
                if string.find(filename, modData.path, nil, true) then
                    return modData
                end
            end
        end
    end
end
function THModManager.getModVersion(self, modName)
    local modData = self:getActiveMod(modName)
    local modVersionText = "0.0.0.0"
    local modVersionData = nil
    if modData ~= nil then
        modVersionData = modData.version
        if modVersionData ~= nil then
            modVersionText = modVersionData.text or modVersionText
        end
    end
    return modVersionText, modVersionData
end
THUtils.pcall(initScript)